/*******

MICE client

copyright 2001, 2002, Alan H. Clifford.

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your option)
any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc., 59
Temple Place, Suite 330, Boston, MA 02111-1307 USA


Alan Clifford can be contacted at mice@clifford.ac

Latest version at http://www.clifford.ac

$Id: mice.client.c,v 2.43.1.4 2002/07/15 22:28:57 alan Exp $
$Name: release4_2 $

 ********/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <signal.h>
#include <sys/stat.h>
#include <getopt.h>
#include <ctype.h>
/*
 #include <fcntl.h>
 */

// #include <syslog.h>

#include "mice.codes.h"


// #define MAXDATASIZE 16 // max number of bytes we can get at once, testing
#define MAXDATASIZE 1024 // max number of bytes we can get at once
#define SENDBUFFSIZE 512 // max number of bytes we can send at once

// for real server
#define MICEPORT 5022 // the port client will be connecting to
#define MICESELECTTIMEOUTSEC 40
#define MICEINETDIDENT "mice"

// for demo server
#define DEMOPORT 1953 // the port client will be connecting to
#define DEMOSELECTTIMEOUTSEC 20
#define DEMOINETDIDENT "mice.demo"



typedef const struct Params {
    unsigned short int port;
    char const * inetdident;
    struct timeval selecttimeout;
} PARAMS;

PARAMS * Paramsp = NULL;  // for real or demo server defaults


char const * const Mice_client_version = "Mundungus Internet Connection Enhancer client\r\n($Revision: 2.43.1.4 $)\r\n($Name: release4_2 $)";
char const * const Copyright = "Copyright 2001, Alan H. Clifford";
char const * const Disclaimer = "MICE client comes with ABSOLUTELY NO WARRANTY";




typedef struct {
    char * host_p;
    char * port_p;
    // long port_l;
    // unsigned short int port_usi;
    long a_minutes_l;
    char * command_p[10];
    // char * message_p;
    char * info_p;
    // char * up_p;
    // char * down_p;
    char * miceid_p;
    char * killid_p;  // id of another client to kill
    int updown;  // 0 unset, 1 = up, 2 = down
    int connect;
    int background;
} STARTUP_ARGS;

/* structure for holding server data */
typedef struct {
    /* send buffer */
    char * sendbuff;  // dynamically allocated
    int sendbuffsize;
    int bytes_sent, bytes_notsent;
} SERVER_DATA;


void do_at_exit(void);

void sigalrm_handler(int s);
int get_options(int argc, char *argv[], STARTUP_ARGS *startup);
// int get_options(int argc, char *argv[], struct startup_arg *startup);
int check_ayt(const char *buf);
void s_queue_cstring(char *s, int servr_fd, SERVER_DATA *server_data_ptr,
                     fd_set *masterptr, fd_set *writemasterptr);
void s_queue_nbytes(char *s, int nbytes, int server_fd, SERVER_DATA *server_data_ptr,
                            fd_set *masterptr, fd_set *writemasterptr);
char *s_queue_alloc(SERVER_DATA * server_data_ptr, int nbytes);
void s_queue_send(int server_fd, SERVER_DATA *server_data_ptr,
                  fd_set *masterptr, fd_set *writemasterptr);
void switch_off_stdout(void);
void switch_off_stderr(void);
void switch_off_stdin(void);
int create_socket(STARTUP_ARGS *startup_p);
// int create_socket(struct startup_arg *startup);
unsigned short int getport(STARTUP_ARGS *startup);
// unsigned short int getport(unsigned short int startup_port);
void do_help(void);



void setparams(int real_or_demo)
{
    // real_or_demo, 1 = real, 2 = demo
    static PARAMS params[2] = {  /* [0] is real server, [1] is demo */
        {MICEPORT, MICEINETDIDENT, {MICESELECTTIMEOUTSEC, 0}},
        {DEMOPORT, DEMOINETDIDENT, {DEMOSELECTTIMEOUTSEC, 0}}
    };
    if (2 == real_or_demo) Paramsp = &params[1]; // demo
    else Paramsp = &params[0]; // real
}




int main(int argc, char *argv[])
{
    fd_set master;   // master file descriptor list
    fd_set read_fds; // temp file descriptor list for select()
    int fdmax;        // maximum file descriptor number
    fd_set writemaster;
    fd_set write_fds;
    int select_rc;
    struct timeval select_tv;
    int server_fd, numbytes;
    SERVER_DATA server_data;

    char * miceid_p = NULL;  // ptr to server id.  Initalize to NULL for testing.

    char recvbuf[MAXDATASIZE];
    char data_to_send[SENDBUFFSIZE];
    // struct startup_arg startup;
    STARTUP_ARGS startup;
    time_t alarm_time = 0; // 0 indicates not in use
    struct sigaction sa;  /* for alarm */
    int background = 0; // 0 means not background
    pid_t fork_pid;

    // initialize server data
    server_data.sendbuff = NULL;
    server_data.sendbuffsize = server_data.bytes_sent = server_data.bytes_notsent = 0;

    // not needed - server buffers until crlf received
    /* make standard out unbuffered */
    setvbuf(stdout, NULL, _IONBF, 0);

    setparams(1);  // Set to real server.  Maybe changed when command line parameters read
    // printf("Params port %hu\n", Paramsp->port);

    // printf("%ld", (long)getpid());

    if (get_options(argc, argv, &startup)) {
        fprintf(stderr, "%s\n", "Bad options");
        // do_help();
        exit(1);
    }

    // printf("Params port %hu\n", Paramsp->port);

    // DON'T do any printfs before here if we are going to fork.

    background = startup.background;
    if ((NULL != startup.info_p)
        || (0 != startup.updown)
        || (0 != startup.connect)
        || (NULL != startup.killid_p))
        background = 1;

    if (0 != background) {
        switch (fork_pid = fork()) {
        case 0:
            // child
            switch_off_stdout();
            switch_off_stderr ();
            switch_off_stdin();
            if (-1 == setsid()) {
                exit(5);
            }
            break;
        case -1: // error
            // perror("daemon fork");
            exit(6);
            break;
        default:
            // in parent
            printf("%ld", (long)fork_pid); // the only print to stdout
            // fprintf(stderr, "child pid = %ld\n", (long)fork_pid);
            _exit(0);
            break;
        }
    }
    else {
        atexit(do_at_exit);
        printf("\n\n%s\n%s\n%s\n\n", Mice_client_version, Copyright, Disclaimer);
    }



    // server_fd = create_socket(&startup);


    sa.sa_handler = sigalrm_handler; // action on alarm
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART || SA_NOMASK;
    if (-1 == sigaction(SIGALRM, &sa, NULL)) {
        perror("sigaction");
        exit(1);
    }


    server_fd = create_socket(&startup);


    FD_ZERO(&master);    // clear the master and temp sets
    FD_ZERO(&read_fds);
    FD_ZERO(&writemaster);
    FD_ZERO(&write_fds);


    // add the socket to the master set
    FD_SET(server_fd, &master);

    // keep track of the biggest file descriptor
    fdmax = server_fd; // so far, it's this one




    if (0 == background) {
        // add standard in
        FD_SET(STDIN_FILENO, &master);
        if (STDIN_FILENO > fdmax) {    // keep track of the maximum
            fdmax = STDIN_FILENO;
        }

        snprintf(data_to_send, SENDBUFFSIZE, "%s\r\n", "status");
        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
    }
    else {
        pid_t mypid;
        uid_t myeuid;
        // donotsend also specifies that this client will be killed when the connection dies
        mypid = getpid();
        myeuid = geteuid();
        snprintf(data_to_send, SENDBUFFSIZE, "\002%s\003\r\n", "donotsend");
        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
    }

    {
        // all clients send owner and id.  Need by -c and -k
        pid_t mypid;
        uid_t myeuid;
        mypid = getpid();
        myeuid = geteuid();
        snprintf(data_to_send, SENDBUFFSIZE,
                 "\002OWNERID\003%lu\r\n\002CLIENTID\003%ld\r\n",
                 (unsigned long)myeuid, (long)mypid);
        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
    }

    

    // if there is a server id environment variable,
    // send it before up, down, info and connect
    if (NULL == (miceid_p = getenv("MICEID"))) {
        // else if there is a server id command line parameter,
        // send it before up, down, info and connect
        if (NULL != startup.miceid_p) miceid_p = startup.miceid_p;
    }

    if (NULL != miceid_p) {
        snprintf(data_to_send, SENDBUFFSIZE, "\002SERVERID\003%s\r\n", miceid_p);
        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
    }


    /*
     else if (0 <= startup.a_minutes_l) {
     // if a persistent request
     // send the request
     snprintf(data_to_send, SENDBUFFSIZE, "Connecting\r\n%s\r\n", "connect");
     printf("%s\n", data_to_send);
     s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
     }
     */

    /*
     // send the -m command line message
     else if (NULL != startup.message_p) {
     // char * prefix_p = "donotsend\r\n";
     char * prefix_p = "";
     char * suffix_p = "\r\nquit\r\n";
     int len;
     // background = 1;
     len = SENDBUFFSIZE - strlen(prefix_p) - strlen(suffix_p) - 1;
     strcpy(data_to_send, prefix_p);
     // message may be too long
     strcat(strncat(data_to_send, startup.message_p, len), suffix_p);
     // snprintf(data_to_send, SENDBUFFSIZE, "8100 %s\r\n", startup.messqage_p);
     // printf("message: %s", data_to_send);
     s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
     }
     */


    if ((NULL != startup.info_p) || (0 != startup.updown)) {
        // status and / or info message
        // don't test for server id above as we need to quit
        if (NULL != miceid_p) {  // not really neccessary as the server checks
            if (NULL != startup.info_p) {
                snprintf(data_to_send, SENDBUFFSIZE, "\002INFO\003%s\r\n", startup.info_p);
                s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
            }
            if (1 == startup.updown) {
                snprintf(data_to_send, SENDBUFFSIZE, "\002%s\003\r\n", "ONLINE");
                s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
            }
            else if (2 == startup.updown) {
                snprintf(data_to_send, SENDBUFFSIZE, "\002%s\003\r\n", "OFFLINE");
                s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
            }
        }
        s_queue_cstring("quit\r\n", server_fd, &server_data, &master, &writemaster);
    }

    else if ((NULL != startup.killid_p)) {
        snprintf(data_to_send, SENDBUFFSIZE, "\002KILLCLIENT\003%s\r\n%s\r\n",
                 startup.killid_p, "quit");
        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
    }


    else {
        // connect
        if (0 != startup.connect) {
            // if (NULL != miceid_p) {  // server doesn't check id for connect
            snprintf(data_to_send, SENDBUFFSIZE, "%s\r\n", "connect");
            s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
            // }
            // even if connect did not occur, leave running so it can be killed
        }
        else {
            // send the command line commands
            int ctr;
            for (ctr = 0; ctr < 10; ++ctr) {
                if (NULL != startup.command_p[ctr]) {
                    snprintf(data_to_send, SENDBUFFSIZE, "%s\r\n", startup.command_p[ctr]);
                    s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
                }
            }
        }
        // always allow the alarm for connect and command line commands
        if (0 < startup.a_minutes_l) {
            /* if a non-persistent request */
            alarm(startup.a_minutes_l * 60);
            alarm_time = time(NULL) + (startup.a_minutes_l * 60);
            /* send the request */
            snprintf(data_to_send, SENDBUFFSIZE, "Running for %ld minute%s\r\n",
                     startup.a_minutes_l, startup.a_minutes_l > 1 ? "s" : "");
            printf("%s\n", data_to_send);
            // s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
        }
    }


    // sleep(5);


    for(;;) {
        read_fds = master; // copy it
        write_fds = writemaster; // copy it
        // select_tv.tv_sec = SELECTTIMEOUTSEC;   /* seconds */
        // select_tv.tv_usec = 0;
        select_tv = Paramsp->selecttimeout;
        if (-1 == (select_rc = select(fdmax+1, &read_fds, &write_fds, NULL, &select_tv))) {
            perror("select");
            fprintf(stderr, "error no. %d\n", errno);
            // if (EINTR != errno) exit(1);
        }
        else if (select_rc == 0) {
            if (0 != alarm_time) {
                printf("\n\n\nExiting in %ld seconds\n\n\n", (long)(alarm_time - time(NULL)));
            }
        }
        else {
            /* do sends here */
            if (FD_ISSET(server_fd, &write_fds)) { /* there is data to send */
                s_queue_send(server_fd, &server_data, &master, &writemaster);
            }
            else if (FD_ISSET(server_fd, &read_fds)) {
                /* line buffering needed in the recv because of character clients
                 like Windows telnet */
                if (0 >= (numbytes = recv(server_fd, recvbuf, MAXDATASIZE-1, MSG_NOSIGNAL))) {
                    if (0 == numbytes) {
                        /* server has gone */
                        printf("%s\n", "server has disappeared");
                    }
                    else {
                        perror("recv");
                        /* exit(1); */
                    }
                    close(server_fd);
                    /* remove from master set */
                    FD_CLR(server_fd, &master);
                    /* update maximum */
                    if (server_fd >= fdmax) --fdmax;
                    exit(1);
                }
                else {
                    recvbuf[numbytes] = '\0';
                    // printf("Received: %s",recvbuf);
                    // printf("%s||",recvbuf);
                    printf("%s",recvbuf);
                    // Check for AYT request
                    if (0 != check_ayt(recvbuf)) {
                        snprintf(data_to_send, SENDBUFFSIZE, "\002%s\003\r\n", "CONSOLECLIENTAYT");
                        s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
                    }
                }
            }
            else if (FD_ISSET(STDIN_FILENO, &read_fds)) {
                char * bufptr;
                numbytes = read(STDIN_FILENO, data_to_send, SENDBUFFSIZE - 3);
                *(data_to_send + numbytes) = '\0';
                /* add the neccessary \r\n for Windows telnet */
                if (NULL != (bufptr = strrchr(data_to_send, '\n'))) {
                    *bufptr = '\0';
                    strcat(data_to_send, "\r\n");
                }


                if (!strncasecmp("EXIT", data_to_send, 4)) {
                    close(server_fd);
                    exit(0);
                }
                else {
                    s_queue_cstring(data_to_send, server_fd, &server_data, &master, &writemaster);
                }

            }
        }
    }


    sleep(5);
    close(server_fd);

    return 0;
}


void do_at_exit(void) {
    printf("\n\n%s\n\n", "bye bye");
    /* sleep(2); */
}



void sigalrm_handler(int s) {
    /* alarm */
    // printf("%s\n","Alarm went off");
    exit(2);
}


/*
int old_check_ayt(const char *buf)
{
    // This is not very good yet because if the receive buffer does
    // not contain whole line, the ayt command may be missed
    //
    int rc = 0;
    if ((NULL != strstr(buf, AYTGENERAL)) || (NULL != strstr(buf, AYTCONSOLE)))
        rc = 1;
    return rc;
    }
    */



int check_ayt(const char *buf)
{
    // ayt can be at beginning of the buffer or after a nl.
    // Or there could be a partial ayt at the end of the buffer.
    // Assumes aytgeneral and aytconsole are defines
    // Assumer buffer is bigger that the aytconsole length + 2
    // and that there are more characers after the ???? code.
    const char * bufptr2;
    static int ayt_len = 0;
    // static char partialbuff[7] = "";  // to be malloced in first time
    static char *partialbuff = NULL;
    int remaininglen;
    const char *laststring = buf;  // may be updated in the search for cr's
    int rc = 0;

    if (NULL == partialbuff) {
        // if (0 == ayt_len) {
        // first time
        int aytc_len = strlen(AYTCONSOLE);
        int aytg_len = strlen(AYTGENERAL);
        ayt_len = (aytc_len > aytg_len ? aytc_len : aytg_len);
        remaininglen = ayt_len + 2;
        // printf("\n\n\n\nsize ayt_len %d \n\n\n\n", ayt_len);
        if (NULL == (partialbuff = malloc(ayt_len + 3))) {
            fprintf(stderr, "\n%s\n", "memory allocation problem in check_ayt()");
            exit(1);
        }

    }

    // The partial buffer should start with a cr have up to nl + ayt_len chars
    if ('\0' != *partialbuff) {
        // The partial buffer could contain a crnl that are from an empty line
        // so if next character is a cr, abandon contents of partial buffer.
        // No, doesn't matter.  If next chr is a cr, then it will be picked up
        // in the while loop phase.
        // if ('\r' == *buf) *partialbuff = '\0';
        remaininglen = ayt_len + 2 - strlen(partialbuff);
        // remaining characters
        strncat(partialbuff, buf, remaininglen);
        // if still not enough characters, fill from next buffer
        // otherwise check and null buffer
        if (strlen(partialbuff) >= (ayt_len + 2)) {
            // enough chars
            bufptr2 = partialbuff + 2;
            // printf("\n\tPartial filled? <%s>\n", bufptr2);
            // check for ayt
            if ((0 == strncmp(bufptr2, AYTGENERAL, ayt_len))
                || (0 == strncmp(bufptr2, AYTCONSOLE, ayt_len))) {
                // found
                rc = 1;
            }
            *partialbuff = '\0';
        }
        // otherwise keep for more characters
    }

    if (0 == rc) {  // we only need to catch one in any buffer as server sets count to zero
        // look for beginning of ayt
        // if the ???? code is at the beginning of the buffer, it should
        // be processed in the partialbuf mechanism. Wrong if proceed by an
        // empty line crnl.  But that is picked up here.
        for (bufptr2 = buf;
             (NULL != (bufptr2 = strchr(bufptr2, '\r')));
             ++bufptr2) {
            laststring = bufptr2;  // save for partial check
            // check for ayt
            if (0 == strncmp("\r\n", bufptr2, 2)) {
                bufptr2 += 2;  // could be on the null byte
                if ((0 == strncmp(bufptr2, AYTGENERAL, ayt_len))
                    || (0 == strncmp(bufptr2, AYTCONSOLE, ayt_len))) {
                    // found
                    rc = 1;
                    // printf("\n\t............%s\n", "found");
                    // sleep(5);
                    break;  // from while
                }
            }
        }
        // bufptr2 now invalid
    }

    // check for partial string
    // find the last cr and if there is one and the length is too short,
    // start the partialbuffer
    if (NULL != (bufptr2 = strrchr(laststring, '\r'))) {
        if (strlen(bufptr2) < (ayt_len + 2)) {
            // potential partial
            strncpy(partialbuff, bufptr2, ayt_len + 2);
            // printf("\n\t%s\n", "Potential partial");
        }
        else *partialbuff = '\0';
    }

    return rc;
}


void s_queue_cstring(char *s, int server_fd, SERVER_DATA *server_data_ptr,
                     fd_set *masterptr, fd_set *writemasterptr)
{
    int slen;
    slen = strlen(s);
    s_queue_nbytes(s, slen, server_fd, server_data_ptr, masterptr, writemasterptr);
}



void s_queue_nbytes(char *s, int nbytes, int server_fd, SERVER_DATA *server_data_ptr,
                            fd_set *masterptr, fd_set *writemasterptr)
{
    char *server_buffp;
    if ((FD_ISSET(server_fd, masterptr))  && (server_fd != STDIN_FILENO)) {
        /* get the servers's data send buffer */
        if (NULL != (server_buffp = s_queue_alloc(server_data_ptr, nbytes))) {
            /* server_buffp points to the client's send buffer */
            memcpy(server_buffp, s, nbytes);
            server_data_ptr->bytes_notsent += nbytes;
            FD_SET(server_fd, writemasterptr); /* add to send master set */
        }
    }
    else  FD_CLR(server_fd, writemasterptr); /* remove from send master set */
}



char *s_queue_alloc(SERVER_DATA * server_data_ptr, int nbytes)
{
    /* return a pointer to the place in the buffer where new data can be added,
     after checking/allocating the send buffer size */
    /* note SERVER_data_ptr->sendbuff could be NULL but OK for realloc. Test
     on the buffer size is sufficient */
    char * newbuffp = NULL;
    if (NULL != server_data_ptr) {
        if (nbytes <= (server_data_ptr->sendbuffsize
                       - server_data_ptr->bytes_sent
                       - server_data_ptr->bytes_notsent)) {
            newbuffp = server_data_ptr->sendbuff
                + server_data_ptr->bytes_sent
                + server_data_ptr->bytes_notsent;
        }
        else if (NULL != (newbuffp = realloc(server_data_ptr->sendbuff,
                                             (server_data_ptr->bytes_sent
                                              + server_data_ptr->bytes_notsent
                                              + nbytes) * sizeof(char)))) {
            server_data_ptr->sendbuff = newbuffp;
            newbuffp = server_data_ptr->sendbuff + server_data_ptr->bytes_sent
                + server_data_ptr->bytes_notsent;
            server_data_ptr->sendbuffsize = (server_data_ptr->bytes_sent
                                             + server_data_ptr->bytes_notsent
                                             + nbytes);
            /* temporary */
            // printf("%s\n", "send queue realloc");
            /* sleep(3); */
        }
        /* server_data_ptr->sendbuff could still be NULL or too small */
    }
    /* return newbuffp. This will be null if realloc failed */
    return newbuffp; /* could be NULL */
}


void s_queue_send(int server_fd, SERVER_DATA *server_data_ptr,
                  fd_set *masterptr, fd_set *writemasterptr)
{
    /* send nbytes */
    int bytes_sent;
    // printf("%s\n", "In s_queue_send");
    if ((FD_ISSET(server_fd, masterptr)) && (server_fd != STDIN_FILENO)
        && (NULL != server_data_ptr)) {
        if (-1 == (bytes_sent = send(server_fd, server_data_ptr->sendbuff
                                     + server_data_ptr->bytes_sent,
                                     server_data_ptr->bytes_notsent, MSG_NOSIGNAL))) {
            perror("s_queue_send");
            /* forget everything */
            server_data_ptr->bytes_sent = server_data_ptr->bytes_notsent = 0;
            FD_CLR(server_fd, writemasterptr); /* remove from send master set */
        }
        else {
            server_data_ptr->bytes_sent += bytes_sent;
            server_data_ptr->bytes_notsent -= bytes_sent;
            /* if bytes_notsent < 0, something is wrong */
            if (0 >= server_data_ptr->bytes_notsent) {
                /* all gone */
                server_data_ptr->bytes_sent = server_data_ptr->bytes_notsent = 0;
                FD_CLR(server_fd, writemasterptr); /* remove from send master set */
            }
        }
    }


    else {
        /* not in the master set, so remove from write master set */
        FD_CLR(server_fd, writemasterptr); /* remove from send master set */
    }
    // printf("%s\n", "End of s_queue_send");
}




void switch_off_stdout(void)
{
    struct stat statbuffer;
    // look for /dev/null
    if (-1 == stat("/dev/null", &statbuffer)) {
        perror("stat devnull");
        close(STDOUT_FILENO);
    }
    else if (statbuffer.st_mode & S_IWUSR) {
        // write mode
        // puts("Can write to /dev/null");
        if (NULL == freopen("/dev/null", "a", stdout)) {
            perror("freeopen null");
            close(STDOUT_FILENO);
        }
        else {
            printf("%s\n", "This should not be seen");
        }
    }
}


void switch_off_stderr(void)
{
    // FILE *mytty;
    struct stat statbuffer;
    // look for /dev/null
    if (-1 == stat("/dev/null", &statbuffer)) {
        perror("stat devnull");
        close(STDERR_FILENO);
    }
    else if (statbuffer.st_mode & S_IWUSR) {
        // write mode
        // puts("Can write to /dev/null");
        // fprintf(stderr, "%s\n", "testing This is stderr");
        if (NULL == freopen("/dev/null", "a", stderr)) {
            perror("freeopen null");
            close(STDERR_FILENO);
        }
        else {
            fprintf(stderr, "%s\n", "This should not be seen");
        }
    }
}

void switch_off_stdin(void)
{
    // FILE *mytty;
    struct stat statbuffer;
    // look for /dev/null
    if (-1 == stat("/dev/null", &statbuffer)) {
        perror("stat devnull");
        close(STDIN_FILENO);
    }
    else if (statbuffer.st_mode & S_IRUSR) {
        // write mode
        // puts("Can write to /dev/null");
        // fprintf(stderr, "%s\n", "testing This is stderr");
        if (NULL == freopen("/dev/null", "r", stdin)) {
            perror("freeopen null");
            close(STDIN_FILENO);
        }
    }
}




// int get_options(int argc, char *argv[], struct startup_arg *startup)
int get_options(int argc, char *argv[], STARTUP_ARGS *startup)

{
    char *a_minutes_p = NULL;
    // char * port_p = NULL;  // initialized for checking
    // int optind_temp, arg_error = 0;
    int arg_error = 0;
    // char *ptr;
    struct option long_options[] = {
        {"help", no_argument, 0, 'h'},
        {"demo", no_argument, 0, 'D'},
        {"version", no_argument, 0, 'V'},
        {"background", no_argument, 0, 'b'},
        {"connect", no_argument, 0, 'c'},
        {"kill", required_argument, 0, 'k'},
        {"port", required_argument, 0, 'p'},
        {"alarm", required_argument, 0, 'a'},
        {"command0", required_argument, 0, '0'},
        {"command1", required_argument, 0, '1'},
        {"command2", required_argument, 0, '2'},
        {"command3", required_argument, 0, '3'},
        {"command4", required_argument, 0, '4'},
        {"command5", required_argument, 0, '5'},
        {"command6", required_argument, 0, '6'},
        {"command7", required_argument, 0, '7'},
        {"command8", required_argument, 0, '8'},
        {"command9", required_argument, 0, '9'},
        // {"message", required_argument, 0, 'm'},
        {"information", required_argument, 0, 'i'},
        {"miceid", required_argument, 0, 'M'},
        {"up", no_argument, 0, 'u'},
        {"down", no_argument, 0, 'd'},
        {0,0,0,0}
    };
    int optchar;

    startup->host_p = NULL;  // initialized for checking
    startup->port_p = NULL;  // initialized for checking
    // startup->port_l = 0;
    // startup->port_usi = 0;  // initalized for checking
    startup->a_minutes_l = -1;
    // startup->message_p = NULL;
    startup->info_p = NULL;
    // startup->up_p = NULL;
    // startup->down_p = NULL;
    startup->miceid_p = NULL;
    startup->killid_p = NULL;
    startup->updown = 0;  // not up or down
    startup->connect = 0;
    startup->background = 0;
    {
        // initialize pointers for commands from command line
        int ctr;
        for (ctr = 0; ctr < 10; ++ctr) startup->command_p[ctr] = NULL;
    }

    while (1) {
        optchar = getopt_long(argc, argv, "hDVbck:p:a:0:1:2:3:4:5:6:7:8:9;i:M:ud", long_options, NULL);

        if (EOF == optchar) break;
        switch (optchar) {
        case 'V' :  // printf("%s %s\n", "option h", optarg);
            printf("\n\n%s\n\n", Mice_client_version);
            exit(0);
            break;
        case 'h' :  // printf("%s %s\n", "option v", optarg);
            do_help();
            break;
        case 'D' :  // printf("%s %s\n", "option D", optarg);
            setparams(2);  // Set to use demo server.
            break;
        case 'b' :  // printf("%s %s\n", "option b", optarg);
            startup->background = 1;
            break;
        case 'c' : // printf("%s %s\n", "option c", optarg);
            startup->connect = 1;
            break;
        case 'k' : // printf("%s %s\n", "option k", optarg);
            startup->killid_p = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case 'p' : // printf("%s %s\n", "option p", optarg);
            startup->port_p = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case 'a' : // printf("%s %s\n", "option a", optarg);
            a_minutes_p = optarg;
            break;
        case '0' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[0] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '1' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[1] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '2' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[2] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '3' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[3] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '4' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[4] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '5' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[5] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '6' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[6] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '7' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[7] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '8' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[8] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
        case '9' : // printf("%s %s\n", "option 1", optarg);
            startup->command_p[9] = optarg;
            /* printf("port: %s\n", port_p); */
            break;
            /* case 'm' : // printf("%s %s\n", "option m", optarg);
             startup->message_p = optarg;
             break;
             */
        case 'M' : // printf("%s %s\n", "option m", optarg);
            startup->miceid_p = optarg;
            break;
        case 'i' : // printf("%s %s\n", "option i", optarg);
            startup->info_p = optarg;
            break;
        case 'u' : // printf("%s %s\n", "option u", optarg);
            startup->updown = 1;
            break;
        case 'd' : // printf("%s %s\n", "option d", optarg);
            startup->updown = 2;
            break;
        case '?' :
            fprintf(stderr, "%s\n", "unknown option character");
            ++arg_error;
            break;
        case ':' :
            fprintf(stderr, "%s\n", "missing parameter");
            ++arg_error;
            break;
        default : fprintf(stderr, "what's this %c\n", optchar);
        ++arg_error;
        break;
        }
    }

    // DEMO mode will be set up at this stage point if specified


    // non-options
    // host is first option
    if ((argc - optind) > 0) {
        startup->host_p = argv[optind];
        if ((argc - optind) > 1) {
            // port is second option
            startup->port_p = argv[optind + 1];  // overrides option -p
        }
    }


    /*
     if ((NULL != startup->up_p) && (NULL != startup->down_p)) {
     ++arg_error;
     printf("%s\n", "-u, -d not allowed together");
     }
     */

    /*
     {
     // message and command line connect together not allowed
     int not_allowed = 0;
     // if (NULL != startup->message_p) ++not_allowed;
     if ((NULL != startup->info_p) || (0 != startup->updown)) ++not_allowed;
     if ((0 != startup->connect) || (NULL != a_minutes_p)) ++not_allowed;
     if (1 < not_allowed) {
     printf("%s\n","(-a or -c) with (-i or -u or -d) not allowed together");
     ++arg_error;
     }
     }
     */



    if (NULL != a_minutes_p) {
        // check it is numeric
        startup->a_minutes_l = strtol(a_minutes_p, NULL, 10);
        if (startup->a_minutes_l < 0) {
            fprintf(stderr, "a minutes: %s\n", a_minutes_p);
            fprintf(stderr, "negative minutes %ld, ignoring\n", startup->a_minutes_l);
            startup->a_minutes_l = -1;
            // ++arg_error;
        }
    }


    if (arg_error) fprintf(stderr, "argument errors = %d\n", arg_error);

    return arg_error;
}



unsigned short int getport(STARTUP_ARGS *startup)
// return port in network byte order
{
    unsigned short int port_ns = 0;  // default
    struct servent *miceent;
    if (NULL == startup->port_p) {
        // no command line port
        if (NULL != (miceent = getservbyname(Paramsp->inetdident, "tcp"))) {
            // look for mice or mice.demo
            port_ns = miceent->s_port;
        }
        else port_ns = htons(Paramsp->port); // use numeric port if inet port not found
    }
    else {
        // check if command line port is numeric
        unsigned long port_ul;
        char *endptr;
        port_ul = strtoul(startup->port_p, &endptr, 10);
        if (('\0' == *endptr) && (USHRT_MAX >= port_ul) && (0 < port_ul))   // valid number
            port_ns = htons((unsigned short int)port_ul);
        else if (NULL != (miceent = getservbyname(startup->port_p, "tcp")))
            port_ns = miceent->s_port;
        // back to host short syntax is ntohs(miceent->s_port)
    }

    return port_ns;  // Could be 0 if bad command line port
}



void do_help(void)
{
    char *prog = "mice.client";

    // printf("\n\n%s\n\n", "There is no help yet.");
    printf("\nusage:  %s [server [port]] [options]\n\n", prog);
    printf("%s\n\t%s\n", "-p #, --port #",
           "Where # is the port number or the port name in /etc/services");
    printf("%s\n\t%s\n", "-a #, --alarm #", "Run for # minutes and exit");
    printf("-D, --demo\n\t%s\n","Use the demonstration server");
    printf("%s\n\t%s\n", "-b, --background", "Set background mode");
    printf("%s\n\t%s\n", "-M ServerId, --miceid ServerId",
           "Send the server id back to the server");
    printf("-------------------\n%s\n\t%s\n", "-# command, --command# command",
           "Send command to server (where # is a number from 0 to 9)");
    printf("-------------------\n%s\n\t%s\n", "-c, --connect",
           "Send connection request and set background mode");
    printf("-------------------\n%s\n\t%s\n", "-k client-id, --kill client_id",
           "Kill another client, in background mode, and exit");
    printf("-------------------\n%s\n\t%s\n", "-i text, --information text",
           "Send information to server, in background mode, and exit");
    printf("%s\n\t%s\n\t%s\n", "-u, --up",
           "Inform the server, in background mode, that the external",
           "connection is up and exit");
    printf("%s\n\t%s\n\t%s\n", "-d, --down",
           "Inform the server, in background mode, that the external",
           "connection is down and exit");
    printf("-------------------\n-V, --version\n\t%s version\n", prog);
    printf("-------------------\n%s\n\t%s\n", "-h, --help","Help information");
    printf("-------------------\nexample:  %s\n", prog);
    printf("example:  %s localhost mice -a 5\n", prog);
    printf("example:  %s -p 5022\n", prog);
    printf("example:  %s --demo\n", prog);
    printf("example:  %s -1 \"connect validisp\" -2 isps\n", prog);
    printf("example:  %s -0 \"connect validisp\" -b\n", prog);
    printf("example:  %s -d -i \"no default route\"\n", prog);

    exit(0);
}


int create_socket(STARTUP_ARGS *startup)
{
    struct hostent *he;
    struct sockaddr_in their_addr; // connector's address information
    int server_fd;
    char *host_p;

    host_p = (NULL != startup->host_p ? startup->host_p : "localhost");


    // if (NULL == (he = gethostbyname(startup->host_p))) {  // get the host info
    if (NULL == (he = gethostbyname(host_p))) {  // get the host info
        perror("gethostbyname");
        exit(1);
    }

    if (-1 == (server_fd = socket(PF_INET, SOCK_STREAM, 0)) ) {
        perror("socket");
        exit(1);
    }


    their_addr.sin_family = AF_INET;    // host byte order
    their_addr.sin_port = getport(startup);  // short, network byte order
    their_addr.sin_addr = *((struct in_addr *)he->h_addr);
    memset(&(their_addr.sin_zero), '\0', 8);  // zero the rest of the struct

    if (-1 == connect(server_fd, (struct sockaddr *)&their_addr,
                sizeof(struct sockaddr))) {
        perror("connect");
        fprintf(stderr, "%s\n", host_p);  // needs port as well
        exit(1);
    }

    {
        struct linger sock_linger;
        int linger_size = sizeof(struct linger);
        if (-1 == getsockopt(server_fd, SOL_SOCKET, SO_LINGER,
                             &sock_linger, &linger_size)) {
            perror("getsockopt");
            // exit(1);
        }
        else {
            // printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);
        }
        if((0 == sock_linger.l_onoff) || (500 > sock_linger.l_linger)) {
            sock_linger.l_onoff = 1;
            sock_linger.l_linger = 500;
            if (-1 == setsockopt(server_fd, SOL_SOCKET, SO_LINGER,
                                 &sock_linger, linger_size)) {
                perror("setsockopt");
                // exit(1);
            }
            else {
                // printf("linger %d:%d\n", sock_linger.l_onoff, sock_linger.l_linger);
            }
        }
    }
    return server_fd;
}

